<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "fertilizer_norms".
 *
 * @property int $id
 * @property string $district Район
 * @property string $culture Культура
 * @property string $n_kg_ha N кг/га
 * @property string $p2o5_kg_ha P2O5 кг/га
 * @property string $k2o_kg_ha K2O кг/га
 * @property string|null $recommendation_basis Основание рекомендации
 * @property int $created_at Дата создания
 * @property int $updated_at Дата обновления
 */
class FertilizerNorms extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%fertilizer_norms}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['district', 'culture', 'n_kg_ha', 'p2o5_kg_ha', 'k2o_kg_ha'], 'required'],
            [['n_kg_ha', 'p2o5_kg_ha', 'k2o_kg_ha'], 'number', 'min' => 0],
            [['recommendation_basis'], 'string'],
            [['created_at', 'updated_at'], 'integer'],
            [['district', 'culture'], 'string', 'max' => 100],
            [['district', 'culture'], 'unique', 'targetAttribute' => ['district', 'culture']],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'district' => 'Район',
            'culture' => 'Культура',
            'n_kg_ha' => 'Азот (N кг/га)',
            'p2o5_kg_ha' => 'Фосфор (P₂O₅ кг/га)',
            'k2o_kg_ha' => 'Калий (K₂O кг/га)',
            'recommendation_basis' => 'Основание рекомендации',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'timestamp' => [
                'class' => \yii\behaviors\TimestampBehavior::className(),
                'createdAtAttribute' => 'created_at',
                'updatedAtAttribute' => 'updated_at',
            ],
        ];
    }

    /**
     * @param string $district
     * @param string $culture
     * @return FertilizerNorms|null
     */
    public static function getNorms($district, $culture)
    {
        return self::find()
            ->where(['district' => $district, 'culture' => $culture])
            ->one();
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getNormsByDistrict($district)
    {
        return self::find()
            ->where(['district' => $district])
            ->indexBy('culture')
            ->all();
    }

    /**
     * @return array
     */
    public function getNormsArray()
    {
        return [
            'n' => (float)$this->n_kg_ha,
            'p2o5' => (float)$this->p2o5_kg_ha,
            'k2o' => (float)$this->k2o_kg_ha,
        ];
    }

    /**
     * @return float
     */
    public function getTotalNorm()
    {
        return (float)$this->n_kg_ha + (float)$this->p2o5_kg_ha + (float)$this->k2o_kg_ha;
    }

    /**
     * @param string $district
     * @param string $culture
     * @param float $area га
     * @return array
     */
    public static function calculateFertilizerCost($district, $culture, $area = 1)
    {
        $norms = self::getNorms($district, $culture);
        if (!$norms) {
            return null;
        }

        $ureaPrice = FertilizerPrices::getPrice($district, 'Мочевина (карбамид)', 'avg');
        $dapPrice = FertilizerPrices::getPrice($district, 'DAP/Диаммофоска 10:26:26', 'avg');
        $potashPrice = FertilizerPrices::getPrice($district, 'Хлористый калий', 'avg') ?? 25000;

        $ureaAmount = $norms->n_kg_ha / 0.46; // кг/га мочевины
        $dapAmount = max($norms->p2o5_ha / 0.46, $norms->n_kg_ha / 0.18) / 1000;
        $potashAmount = $norms->k2o_kg_ha / 0.60 / 1000;

        $costs = [
            'n' => $ureaAmount * $ureaPrice * $area,
            'p2o5' => $dapAmount * $dapPrice * $area,
            'k2o' => $potashAmount * $potashPrice * $area,
        ];

        $costs['total'] = array_sum($costs);

        return [
            'norms' => $norms->getNormsArray(),
            'costs' => $costs,
            'area' => $area,
            'total_cost_per_ha' => $costs['total'] / $area,
        ];
    }

    /**
     * @param string $district
     * @return array
     */
    public static function getCulturesForDistrict($district)
    {
        $norms = self::find()
            ->where(['district' => $district])
            ->all();

        $result = [];
        foreach ($norms as $norm) {
            $result[$norm->culture] = $norm->culture;
        }

        return $result;
    }

    /**
     * @return array
     */
    public static function getAllCultures()
    {
        return self::find()
            ->select('culture')
            ->distinct()
            ->orderBy('culture')
            ->column();
    }

    /**
     * @return array
     */
    public static function getDistrictsList()
    {
        return self::find()
            ->select('district')
            ->distinct()
            ->indexBy('district')
            ->column();
    }

    /**
     * @param string $district
     * @return \yii\db\ActiveQuery
     */
    public static function findByDistrict($district)
    {
        return self::find()->where(['district' => $district]);
    }

    /**
     * @return string
     */
    public function getFormattedNorms()
    {
        return "N: {$this->n_kg_ha} кг/га, P₂O₅: {$this->p2o5_kg_ha} кг/га, K₂O: {$this->k2o_kg_ha} кг/га";
    }
    /**
     * @param string $district
     * @return array
     */

}